"""httpx Auth implementation for Keycloak OIDC Device Flow."""


from typing import Callable, Optional
from requests.auth import AuthBase
from requests.models import PreparedRequest

from acmadauth.tokenset import TokenSet

from ..handlers import KeycloakOIDCDeviceFlowHandler
from ..tokencaches.base import TokenCache


class RequestsDeviceFlowAuth(AuthBase):
    """
    requests auth module that injects Bearer tokens and refreshes / device-logins as needed.

    Args:
        flow (KeycloakOIDCDeviceFlowHandler): The device flow handler to use.
        cache (TokenCache): The token cache to use.
        open_browser (bool): Whether to open the browser automatically during device flow.
        show_qr (bool): Whether to display a QR code for the device flow URL.
        show_url (bool): Whether to display the device flow URL.
        login_url_callback (Optional[Callable[[str], None]]): Optional callback function to receive

    """

    def __init__(
        self,
        flow: KeycloakOIDCDeviceFlowHandler,
        cache: TokenCache,
        *,
        open_browser: bool = False,
        show_qr: bool = True,
        show_url: bool = True,
        login_url_callback: Optional[Callable[[str], None]] = None,
    ) -> None:
        self.flow = flow
        self.cache = cache
        self.open_browser = open_browser
        self.show_qr = show_qr
        self.show_url = show_url
        self.login_url_callback = login_url_callback

    def __call__(self, request: PreparedRequest) -> PreparedRequest:
        """
        The auth flow that injects tokens into requests. Automatically refreshes or runs device flow as needed.

        Args:
            request: The outgoing HTTP request.

        Returns:
            The modified request with Authorization header.
        """
        tokens: TokenSet = self.flow.ensure_tokens(
            self.cache, open_browser=self.open_browser, show_qr=self.show_qr, show_url=self.show_url, login_url_callback=self.login_url_callback
        )
        request.headers["Authorization"] = (
            f"{tokens.token_type} {tokens.access_token}"
        )
        return request
