import time

import httpx

from acmadauth import handlers
from acmadauth.handlers import KeycloakOIDCDeviceFlowHandler
from acmadauth.tokencaches.memory import MemoryTokenCache
from acmadauth.tokenset import JWTToken, TokenSet


class _FakeResponse:
    def __init__(self, status_code=200, payload=None):
        self.status_code = status_code
        self._payload = payload or {}
        self.text = str(self._payload)

    def json(self):
        return self._payload


def test_start_device_authorization_confidential_client_uses_basic_auth(monkeypatch):
    calls = []

    class FakeClient:
        def __init__(self, *args, **kwargs):
            pass

        def __enter__(self):
            return self

        def __exit__(self, exc_type, exc, tb):
            return False

        def post(self, url, data=None, auth=None):
            calls.append({"url": url, "data": data, "auth": auth})
            return _FakeResponse(payload={"device_code": "d"})

    handler = KeycloakOIDCDeviceFlowHandler(
        realm_url="https://example.test",
        client_id="client",
        client_secret="secret",
    )
    monkeypatch.setattr(handler, "_get_discovery", lambda: {"device_authorization_endpoint": "https://auth/device"})
    monkeypatch.setattr(handlers.httpx, "Client", FakeClient)

    handler.start_device_authorization()

    assert len(calls) == 1
    assert calls[0]["auth"] is not httpx.USE_CLIENT_DEFAULT
    assert isinstance(calls[0]["auth"], httpx.BasicAuth)
    assert calls[0]["data"]["client_id"] == "client"
    assert "client_secret" not in calls[0]["data"]


def test_start_device_authorization_public_client_no_auth(monkeypatch):
    calls = []

    class FakeClient:
        def __init__(self, *args, **kwargs):
            pass

        def __enter__(self):
            return self

        def __exit__(self, exc_type, exc, tb):
            return False

        def post(self, url, data=None, auth=None):
            calls.append({"url": url, "data": data, "auth": auth})
            return _FakeResponse(payload={"device_code": "d"})

    handler = KeycloakOIDCDeviceFlowHandler(
        realm_url="https://example.test",
        client_id="client",
        client_secret=None,
    )
    monkeypatch.setattr(handler, "_get_discovery", lambda: {"device_authorization_endpoint": "https://auth/device"})
    monkeypatch.setattr(handlers.httpx, "Client", FakeClient)

    handler.start_device_authorization()

    assert len(calls) == 1
    assert calls[0]["auth"] is httpx.USE_CLIENT_DEFAULT
    assert calls[0]["data"]["client_id"] == "client"
    assert "client_secret" not in calls[0]["data"]


def test_ensure_tokens_refresh_uses_refresh_token_string(monkeypatch):
    handler = KeycloakOIDCDeviceFlowHandler(
        realm_url="https://example.test",
        client_id="client",
    )

    captured = {}

    def fake_refresh(refresh_token):
        captured["token"] = refresh_token
        return TokenSet(
            access_token=JWTToken(token="new-access", expires_in=3600),
            refresh_token=None,
            token_type="Bearer",
        )

    monkeypatch.setattr(handler, "refresh", fake_refresh)

    expired_access = JWTToken(token="access", expires_in=1, time_obtained=time.time() - 10)
    valid_refresh = JWTToken(token="refresh-token", expires_in=3600, time_obtained=time.time())
    cache = MemoryTokenCache()
    cache.save(TokenSet(access_token=expired_access, refresh_token=valid_refresh, token_type="Bearer"))

    new_tokens = handler.ensure_tokens(cache)

    assert captured["token"] == "refresh-token"
    assert cache.load() == new_tokens
