import json
import logging
from typing import Optional

import keyring
import keyring.backend
import keyring.errors

from ..errors import (TokenCacheClearError, TokenCacheLoadError,
                      TokenCacheSaveError)
from ..tokenset import TokenSet
from .base import TokenCache

logger = logging.getLogger(__name__)


class KeyringTokenCache(TokenCache):
    """
    A secure token cache implementation using the system keyring. It uses the python-keyring library
    to store and retrieve tokens securely. This implementation is suitable for production use.
    By default, it uses the system's default keyring backend, but a custom backend can be provided.
    See https://pypi.org/project/keyring/ for more details on keyring backends.

    Args:
        service_name: The service name to use for storing tokens in the keyring (e.g., ``myapp:oidc:<issuer-host-or-hash>:<client_id>``).
        keyring_backend (Optional[keyring.backend.KeyringBackend]): Optional custom keyring backend to use.
    """

    # service: myapp:oidc:<issuer-host-or-hash>:<client_id>

    def __init__(self, service_name: str,
                 keyring_backend: Optional[keyring.backend.KeyringBackend] = None) -> None:
        self.service_name = service_name
        self.keyring_backend = keyring_backend
        if keyring_backend:
            keyring.set_keyring(keyring_backend)

    def load(self) -> Optional[TokenSet]:
        try:
            data = keyring.get_password(self.service_name, "token_set")
            if data is None:
                return None
            return TokenSet.from_json(json.loads(data))
        except (keyring.errors.InitError, keyring.errors.NoKeyringError, keyring.errors.KeyringLocked) as e:
            raise TokenCacheLoadError(f"Failed to load token cache: {str(e)}")
        except (json.JSONDecodeError) as e:
            raise TokenCacheLoadError(
                f"Failed to decode token cache data: {str(e)}"
            )

    def save(self, token_set: TokenSet) -> None:
        try:
            keyring.set_password(
                self.service_name, "token_set", json.dumps(
                    token_set.to_json()))
        except keyring.errors.KeyringError as e:
            raise TokenCacheSaveError(f"Failed to save token cache: {str(e)}")

    def clear(self) -> None:
        try:
            keyring.delete_password(self.service_name, "token_set")
        except keyring.errors.PasswordDeleteError as e:
            raise TokenCacheClearError(
                f"Failed to clear token cache: {str(e)}")


# class TestKeyring(keyring.backend.KeyringBackend):
#     """A test keyring which always outputs the same password
#     """
#     priority = 1

#     def __init__(self):
#         super().__init__()
#         self.__keyring = {}

#     def set_password(self, servicename, username, password):
#         self.__keyring[(servicename, username)] = password

#     def get_password(self, servicename, username):
#         return self.__keyring.get((servicename, username), None)

#     def delete_password(self, servicename, username):
#         try:
#             del self.__keyring[(servicename, username)]
#         except KeyError:
#             raise keyring.errors.PasswordDeleteError("Password not found")
