import json
import warnings
from typing import Optional

from ..errors import (TokenCacheClearError, TokenCacheLoadError,
                      TokenCacheSaveError)
from ..tokenset import TokenSet
from .base import TokenCache


class JsonFileTokenCache(TokenCache):
    """
    A simple JSON file-based token cache for testing purposes only.

    .. danger::
        Do not use this cache in production environments!
        This implementation stores tokens in plain text in a local file.
        Consider using :class:`KeyringTokenCache <acmadauth.tokencaches.KeyringTokenCache>` for secure storage.
        If you need a secure file-based cache, :class:`KeyringTokenCache <acmadauth.tokencaches.KeyringTokenCache>`
        can be used with `keyrings.cryptfile <https://pypi.org/project/keyrings.cryptfile/>`_.

    Args:
        file_path (str): Path to the JSON file to use for caching tokens.
    """

    def __init__(self, file_path: str) -> None:
        warnings.warn(
            "JsonFileTokenCache is an insecure token cache for testing purposes only. "
            "Do not use in production environments. Consider using KeyringTokenCache instead.",
            UserWarning,
        )
        self.file_path = file_path

    def load(self) -> Optional[TokenSet]:
        try:
            with open(self.file_path, 'r') as f:
                data = json.load(f)
                return TokenSet.from_json(data)
        except FileNotFoundError:
            return None
        except json.JSONDecodeError as e:
            raise TokenCacheLoadError(f"Failed to load token cache: {str(e)}")

    def save(self, token_set: TokenSet) -> None:
        try:
            import os
            fd = os.open(self.file_path, os.O_WRONLY |
                         os.O_CREAT | os.O_TRUNC, 0o600)
            with os.fdopen(fd, 'w') as f:
                json.dump(token_set.to_json(), f)
        except Exception as e:
            raise TokenCacheSaveError(f"Failed to save token cache: {str(e)}")

    def clear(self) -> None:
        try:
            import os
            os.remove(self.file_path)
        except FileNotFoundError:
            return
        except Exception as e:
            raise TokenCacheClearError(
                f"Failed to clear token cache: {str(e)}")
